﻿package {
	
	import flash.display.Shader;
	import flash.display.SimpleButton;
	import flash.display.Shape;
	import flash.display.Sprite;
	import flash.events.Event;
	import flash.events.KeyboardEvent;
	import flash.events.MouseEvent;
	import flash.display.MovieClip;
	import flash.events.TimerEvent;
	import flash.geom.Point;
	import flash.ui.Keyboard;
	import flash.utils.Timer;
	
	
	/**
	 *  1. フィールドに壁を描画する
	 *  2. フィールドにテトリミノを表示してみる
	 *  3. テトリミノを動かしてみる
	 *  4. テトリミノの回転
	 *  5. 回転の補正
	 *  6. テトリミノを固定する
	 *  7. ラインを消す
	 *  8. システム1（ゲームスタート、ゲームオーバー、ボタンクラスの機能拡張）
	 *  9. システム2（テトリミノ自動落下機能追加）
	 * 10. 次のテトリミノ
	 * 11. ハードドロップ
	 * 12. ゴーストブロック
	 * 
	 * @author necoEngine bamboo-satellites
	 * @version 0.2.1
	 * @see http://www.bamboo-satellites.com/
	 * 
	 * コードブロック
	 * Constructor
	 * - Init
	 * - System
	 * - Tetromino Create
	 * - - Next Tetromino
	 * - Tetromino Control
	 * - - Data Control Tools
	 * - - Tetromino Hit Check
	 * - Field Operater
	 * - - Field Search Tools
	 */
	public class main extends MovieClip {
		
		private const CELL_SIZE:uint      = 20;   // セルのサイズ。正方形
		private const FIELD_LEN_W:uint    = 12;   // フィールドの完全幅。壁を除くと10
		private const FIELD_LEN_H:uint    = 23;   // フィールドの完全高さ。壁、テトリミノ出現領域分を除くと20
		private const NUM_TETROMINOS:uint = 7;    // テトリミノの種類数
		
		private const NUM_NEXT:uint     = 5;      // 「次のテトリミノ」の出現予定保存数
		private const NEXT_W:uint       = 120;    // 「次のテトリミノ」表示コンテナの幅
		private const NEXT_H:uint       = 170;    // 「次のテトリミノ」表示コンテナの高さ
		private const LOC_INTERVAL:uint = 60;     // 「次のテトリミノ」の表示間隔
		
		private const LINE_COLOR:uint = 0x999999; // セルの線の色
		private const FILL_RAG:uint   = 120;      // ラインを消すときのタイムラグ
		private const END_DELEY:uint  = 30;       // ゲームオーバー時の描画間隔
		
		private const DROP_MAX:uint   = 1000;     // テトリミノが落下する速度の最低値（deleyプロパティの最大値）
		private const DROP_MIN:uint   = 50;       // テトリミノが落下する速度の最高値（deleyプロパティの最小値）
		private const ACCEL_MAX:uint  = 10;       // テトリミノが落下する速度の加速度上限（dropAccel最大値）
		private const BOOST:uint      = 8;        // dropCountの上限
		
		private const // フィールドデータで使用するコード
			EMPTY:uint      = 0,   // 空白コード
			TETROMINO1:uint = 1,   // テトリミノコード 棒形
			TETROMINO2:uint = 2,   // テトリミノコード　正方形
			TETROMINO3:uint = 3,   // テトリミノコード　S字
			TETROMINO4:uint = 4,   // テトリミノコード　Z字
			TETROMINO5:uint = 5,   // テトリミノコード　J字
			TETROMINO6:uint = 6,   // テトリミノコード　L字
			TETROMINO7:uint = 7,   // テトリミノコード　T字
			GHOST:uint      = 9,   // ゴーストブロックコード
			FIXED:uint      = 10,  // テトリミノ固定コード
			WALL:uint       = 99,  // 壁コード
			GRAY_OUT:uint   = 100; // ゲームオーバー演出用コード
		
		private const // 描画メソッドに使用する描画タイプ指示コード
			DROP:uint  = 0, // 落下中コード
			CLEAR:uint = 1, // 削除指示コード
			FILL:uint  = 2; // ラインを更新する指示コード
		
		private const GAME_START:uint  = 0x00000001; // 「次のテトリミノ」の順番待ち（処理中）
		private const WAITING:uint     = 0x00000002; // 「次のテトリミノ」の順番待ち（処理中）
		private const HARDDROP_ON:uint = 0x00000004; // ハードドロップ実行
		private const GHOST_ON:uint    = 0x00000008; // ゴーストブロック作動
		private const GAME_OVER:uint   = 0x80000000; // ゲームオーバー
		
		
		private var systemStatus:uint;             // システムステータス。ビットマスクフラグ
		private var player1session:DataStructure;  // データ構造体
		
		private var nextCodes:Vector.<uint>;       // これから落ちてくる予定のテトリミノのコードを保存
		private var nextBoard:Sprite;              // 次のテトリミノの表示コンテナ
		private var drawBoard:Shape;               // 次のテトリミノを描画する描画オブジェクト
		private var drawMask:Shape;                // drawBoardのマスク
		private var drawLocation:int;              // 次のテトリミノを描画する際の描画座標
		private var spools:Vector.<DataStructure>; // 次のテトリミノを待っているセッションを保存
		
		private var startButton:ButtonCreater;     // スタートボタンインスタンス
		private var ghostButton:ButtonCreater;     // ゴーストブロック機能のオン/オフ切替
		
		
		public function main():void {
			
			if (stage) init();
			else addEventListener(Event.ADDED_TO_STAGE, init);
			
		} // constructor close
		
		
		
//================================================================================
// Init
//
// データの初期化関連の処理
//================================================================================
		private function init(e:Event = null):void {
			
			removeEventListener(Event.ADDED_TO_STAGE, init);
			// entry point
			
			systemStatus = 0;
			
			// プレイデータ構造体。プレイヤー1のデータ。
			player1session = new DataStructure();
			
			// フィールドデータの初期化
			player1session._field = fieldInit();
			
			// フィールドを描画する表示コンテナの設定
			player1session._container = new Shape();
			addChild(player1session._container);
			player1session._container.x = CELL_SIZE;
			player1session._container.y = CELL_SIZE;
			drawField(player1session); // フィールド描画。壁だけを描画
			
			// 次のテトリミノ表示コンテナ設置
			drawBoardInit();
			
			//スタートボタンインスタンス設定
			startButton = new ButtonCreater(" S T A R T ");
			addChild(startButton);
			startButton.x = stage.stageWidth  - startButton.width  - CELL_SIZE;
			startButton.y = stage.stageHeight - startButton.height - CELL_SIZE;
			startButton.addEventListener(MouseEvent.MOUSE_DOWN, startHandler);
			
			//ゴーストブロックインスタンス設定
			ghostButton = new ButtonCreater(" GHOST ON ");
			addChild(ghostButton);
			ghostButton.x = stage.stageWidth - ghostButton.width  - CELL_SIZE;
			ghostButton.y = startButton.y    - ghostButton.height - 10;
			ghostButton.addEventListener(MouseEvent.MOUSE_DOWN, ghostHandler);
			systemStatus |= GHOST_ON;
			
		} // init() close
		
		
		private function fieldInit():Vector.<Vector.<int>> {
			/**
			 * 機能：フィールドの初期化データ配列を返す
			 * 引数：なし
			 * 戻り値：Vector2次元配列
			 * 備考：最終的には初期描画処理も含める予定
			 */
			
			return Vector.<Vector.<int>>([
				Vector.<int>([ WALL, WALL, WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL, WALL, WALL]),
				Vector.<int>([ WALL, WALL, WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL, WALL, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY, WALL]),
				Vector.<int>([ WALL, WALL, WALL, WALL, WALL, WALL, WALL, WALL, WALL, WALL, WALL, WALL])
			]);
			
		} // fieldInit() close
		
		private function tetriminoInit(code:uint):Vector.<Vector.<uint>> {
			/**
			 * 機能：引数で取ったテトリミノコードに対応したテトリミノのパターン配列を返す
			 * 引数：uint テトリミノのタイプを表すコード
			 * 戻り値：Vector2次元配列　テトリミノのパターンデータ。
			 * 備考：
			 */
			
			switch (code) {
				case TETROMINO1: // 棒形
					return Vector.<Vector.<uint>>([
						Vector.<uint>([EMPTY,     EMPTY,     EMPTY,     EMPTY]),
						Vector.<uint>([TETROMINO1,TETROMINO1,TETROMINO1,TETROMINO1]),
						Vector.<uint>([EMPTY,     EMPTY,     EMPTY,     EMPTY]),
						Vector.<uint>([EMPTY,     EMPTY,     EMPTY,     EMPTY])
					]);
				case TETROMINO2: // 正方形
					return Vector.<Vector.<uint>>([
						Vector.<uint>([TETROMINO2, TETROMINO2]),
						Vector.<uint>([TETROMINO2, TETROMINO2])
					]);
				case TETROMINO3: // S字
					return Vector.<Vector.<uint>>([
						Vector.<uint>([EMPTY,      TETROMINO3, TETROMINO3]),
						Vector.<uint>([TETROMINO3, TETROMINO3, EMPTY]),
						Vector.<uint>([EMPTY,      EMPTY,      EMPTY])
					]);
				case TETROMINO4: // Z字
					return Vector.<Vector.<uint>>([
						Vector.<uint>([TETROMINO4, TETROMINO4, EMPTY]),
						Vector.<uint>([EMPTY,      TETROMINO4, TETROMINO4]),
						Vector.<uint>([EMPTY,      EMPTY,      EMPTY])
					]);
				case TETROMINO5: // J字
					return Vector.<Vector.<uint>>([
						Vector.<uint>([TETROMINO5, EMPTY,      EMPTY]),
						Vector.<uint>([TETROMINO5, TETROMINO5, TETROMINO5]),
						Vector.<uint>([EMPTY,      EMPTY,      EMPTY])
					]);
				case TETROMINO6: // L字
					return Vector.<Vector.<uint>>([
						Vector.<uint>([EMPTY,      EMPTY,      TETROMINO6]),
						Vector.<uint>([TETROMINO6, TETROMINO6, TETROMINO6]),
						Vector.<uint>([EMPTY,      EMPTY,      EMPTY])
					]);
				default: // T字
					return Vector.<Vector.<uint>>([
						Vector.<uint>([EMPTY,      TETROMINO7, EMPTY]),
						Vector.<uint>([TETROMINO7, TETROMINO7, TETROMINO7]),
						Vector.<uint>([EMPTY,      EMPTY,      EMPTY])
					]);
			}
			
		} // tetriminoInit() close
		
		private function colorInit(code:uint):uint {
			/**
			 * 機能：引数で取ったコードに対応したカラー値を返す
			 * 引数：uint
			 * 戻り値：uint 16進数
			 * 備考：
			 */
			
			switch (code) {
				case TETROMINO1: return 0x4DE6E1; // 棒形
				case TETROMINO2: return 0xE9EE11; // 正方形
				case TETROMINO3: return 0x46DF20; // S字
				case TETROMINO4: return 0xED2212; // Z字
				case TETROMINO5: return 0x2746D8; // J字
				case TETROMINO6: return 0xE49E1B; // L字
				case TETROMINO7: return 0xEA68E7; // T字
				case GHOST:      return 0x999999; // ゴーストブロックカラー
				case GRAY_OUT:   return 0x666666; // ゲームオーバー描画カラー
				default:         return 0x7A5B52; // 壁
			}
			
		} // colorInit() close
		
		private function pointInit(code:uint):uint {
			/**
			 * 機能：引数で取ったコードに対応した位置座標の初期配置x値を返す
			 * 引数：uint
			 * 戻り値：uint
			 * 備考：
			 */
			
			switch (code) {
				case TETROMINO2: return 5;
				case TETROMINO4: return 5;
				case TETROMINO6: return 5;
				default:         return 4;
			}
			
		} // pointInit() close
		
		private function drawBoardInit():void {
			/**
			 * 機能：出現予定テトリミノを表示するオブジェクトの初期設定
			 * 引数：なし
			 * 戻り値：なし
			 * 備考：
			 */
			
			// 次のテトリミノ表示コンテナ
			nextBoard = new Sprite();
			addChild(nextBoard);
			nextBoard.graphics.beginFill(LINE_COLOR);
			nextBoard.graphics.drawRoundRect(-NEXT_W * 0.5, 0, NEXT_W, NEXT_H, 10);
			nextBoard.graphics.endFill();
			nextBoard.x = stage.stageWidth - nextBoard.width / 2 - CELL_SIZE;
			nextBoard.y = CELL_SIZE;
			
			// 次のテトリミノ描画オブジェクト
			drawBoard = new Shape();
			nextBoard.addChild(drawBoard);
			
			// マスク
			drawMask = new Shape();
			nextBoard.addChild(drawMask);
			drawMask.graphics.beginFill(LINE_COLOR);
			drawMask.graphics.drawRoundRect(-NEXT_W * 0.5, 0, NEXT_W, NEXT_H, 10);
			drawMask.graphics.endFill();
			drawBoard.mask = drawMask;
			
		} // drawBoardInit() close
		
		private function nextCodesInit(num:uint):Vector.<uint> {
			/**
			 * 機能：次のテトリミノ用に出現予定テトリミノコードを指定数分の配列で用意する。
			 * 引数：用意する予定数
			 * 戻り値：配列で用意したテトリミノコード
			 * 備考：
			 */
			
			var i:uint, program:Vector.<uint>;
			program = new Vector.<uint>(num);
			for (i = 0; i < num; i++) program[i] = Math.floor(Math.random() * 7) + 1;
			return program;
			
		} // nextCodesInit() close
		
		
		
//================================================================================
// System
//
// 主にゲームを運行して行く為の大枠になる処理
//================================================================================
		private function startHandler(event:MouseEvent):void {
			/**
			 * 機能：ゲームスタート処理を行う。スタートボタンインスタンスのリスナー関数
			 * 引数：イベントオブジェクト
			 * 戻り値：なし
			 * 備考：
			 */
			
			//スタートボタンのイベントリスナー削除
			startButton.removeEventListener(MouseEvent.MOUSE_DOWN, startHandler);
			startButton.activeShifter(false);
			
			systemStatus |= GAME_START;
			systemStatus &= ~GAME_OVER;
			spools = new Vector.<DataStructure>();
			
			// テトリミノデータを用意してフィールドデータに反映して描画
			player1session._tetromino = new Vector.<Vector.<uint>>(4);
			player1session._location  = new Point(0, 0);
			nextCodes = nextCodesInit(NUM_NEXT); // 出現予定テトリミノコード生成
			setTetromino(player1session, true);  // 最初のテトリミノを配置
			
			// テトリミノ自動落下タイマー設定
			dropTimerSet(player1session);
			
			// キーボードイベント設定
			keyHandlerSet(player1session)
			
		} // startHandler() close
		
		
		private function onGameOver(data:DataStructure):void {
			/**
			 * 機能：ゲームオーバー用描画のループ処理を行う。
			 * 引数：第1引数 プレイヤーデータ構造体
			 * 戻り値：なし
			 * 備考：具体的にはゲームオーバー時用のフィールドマッピングと描画実行、スタートボタンのリスナー関数登録
			 */
			
			var x:int, y:int, index:int = FIELD_LEN_H - 1, mappingTimer:Timer;
			var field:Vector.<Vector.<int>> = data._field;
			
			mappingTimer = new Timer(END_DELEY);
			mappingTimer.addEventListener(TimerEvent.TIMER,
				function (event:TimerEvent):void { //　無名リスナー関数の登録
					
					for (x = 0; x < FIELD_LEN_W; x++) {
						if (field[index][x] > EMPTY) field[index][x] = GRAY_OUT;
						drawField(data);
					}
					
					index--;
					
					if (index < 0) {
						mappingTimer.stop();
						mappingTimer.removeEventListener(TimerEvent.TIMER, arguments.callee);
						
						data.dataClear();          // データのクリア
						data._field = fieldInit(); // フィールドデータの初期化
						
						drawBoard.graphics.clear(); // 出現予定テトリミノクリア
						nextCodes = null;
						
						// スタートボタンインスタンスに再スタートのリスナー関数を登録する。
						startButton.addEventListener(MouseEvent.MOUSE_DOWN, startHandler);
						startButton.activeShifter();
						
						mappingTimer = null;
					}
				}
			);
			mappingTimer.start();
			
		} // onGameOver() close
		
		
		private function dropTimerSet(data:DataStructure):void {
			/**
			 * 機能：一定時間毎にテトリミノを下へ移動するためのタイマーをセッション毎に設定する。
			 * 引数：データ構造体
			 * 戻り値：なし
			 * 備考：リスナー関数をクロージャにする事で、データ構造体を保持させつつ、そのデータ構造体自体のプロパティにする
			 */
			
			data._dropAccel = 1;
			data._timer = new Timer(DROP_MAX);
			data._dropHandler = function(event:TimerEvent):void { tetrominoControl(Keyboard.DOWN, data); }
			data._timer.addEventListener(TimerEvent.TIMER, data._dropHandler);
			data._timer.start();
			
		} // dropTimerHandler() close
		
		
		private function dropTimerRegulator(data:DataStructure, slow:Boolean = false):void {
			/**
			 * 機能：テトリミノの自動落下の時間間隔を調整する。
			 * 引数：第1引数 データ構造体、第2引数 減速指定かどうかの指示。減速処理なら真。
			 * 戻り値：なし
			 * 備考：通常はテトリミノが一つ固定される度に自動落下の時間間隔値を減算していく（落下が早くなる）。
			 *      時間間隔値から減算する値（accel）は、固定されたテトリミノが一定数に到達すると増加して行く。
			 *      テトリスで消した場合は自動落下の時間間隔値が増加（落下が遅くなる）、減算値は減少する。
			 */
			var timer:Timer  = data._timer;     // 自動落下用のタイマーオブジェクト
			var accel:int    = data._dropAccel; // 加速値
			var counter:uint = data._dropCount; // 固定テトリミノカウンタ
			
			if (slow) {
				// 加速度の減少
				accel -= 2;
				if (accel < 1) accel = 1;
				
				// 落下間隔を増加
				timer.delay *= 2;
				if (timer.delay > DROP_MAX) timer.delay = DROP_MAX;
				
			} else {
				// 加速度の処理
				if (counter >= BOOST) {
					counter = 0; // リセット
					if (accel < ACCEL_MAX) accel++;
				} else {
					counter++; // 固定したテトリミノをカウント
				}
				
				// 自動落下速度の処理
				if (timer.delay < DROP_MIN) timer.delay = DROP_MIN;
				else timer.delay -= accel * 2.5;
			}
			
		} // dropTimerRegulator() close
		
		
		private function ghostHandler(event:MouseEvent):void {
			/**
			 * 機能：ゴーストブロック機能のオン/オフを切り替える。
			 * 引数：イベントオブジェクト
			 * 戻り値：
			 * 備考：切り替え処理はゲームスタートする前に限定
			 */
			
			if (!(systemStatus & GAME_START)) {
				if (systemStatus & GHOST_ON) {
					systemStatus &= ~GHOST_ON;
					ghostButton.onOffShifter(false);
				} else {
					systemStatus |= GHOST_ON;
					ghostButton.onOffShifter();
				}
			}
			
		} // ghostHandler() close
		
		
		
//================================================================================
// Tetromino Create
//
// 主に新しいテトリミノパターンデータの作成にかかわる処理
//================================================================================
		private function setTetromino(data:DataStructure, first:Boolean = false):Boolean {
			/**
			 * 機能：新しいテトリミノパターンデータを用意する（フィールドデータには反映せずに用意するところまで）。
			 * 引数：第1引数 セッションデータ構造体、第2引数 一番最初の配置かどうかの真偽値
			 *      第2引数について 真の場合一番最初として扱う。省略可で、省略した場合は一番最初ではない扱い。
			 * 戻り値：なし
			 * 備考：落下中のテトリミノパターンデータ配列長を新しく用意したパターンデータ配列長とそろえる必要がある
			 */
			
			var x:uint, y:uint, len:uint, gap:int, code:int, patternBuff:Vector.<Vector.<uint>>;
			var field:Vector.<Vector.<int>>      = data._field;     // フィールドデータ
			var tetromino:Vector.<Vector.<uint>> = data._tetromino; // テトリミノパターンデータ
			var loc:Point                        = data._location;  // テトリミノの座標
			
			code = nextCodes[0];               // 新しいパターンコード
			patternBuff = tetriminoInit(code); // 新しいパターンデータ一時保存
			len = patternBuff.length;          // 新しいパターンデータ配列長
			
			gap = tetromino.length - len; // 現在のdropTetrominoの一次元目の配列長との差
			if (gap > 0) for (y = gap;      y > 0; y--) tetromino.pop();                        // 一次元目減少
			if (gap < 0) for (y = gap * -1; y > 0; y--) tetromino.push(new Vector.<uint>(len)); // 一次元目増加
			
			for (y = 0; y < len; y++) {
				if (first) {
					tetromino[y] = new Vector.<uint>(len); // フィールドデータ初期化時の一番最初の実行
				} else {
					gap = tetromino[y].length - len; // 現在のdropTetrominoの二次元目の配列長との差
					if (gap > 0) for (x = gap;      x > 0; x--) tetromino[y].pop();   // 二次元目減少
					if (gap < 0) for (x = gap * -1; x > 0; x--) tetromino[y].push(0); // 二次元目増加
				}
				
				tetromino[y] = patternBuff[y].concat();
			}
			
			// 初期配置座標設定
			loc.x = pointInit(code);
			loc.y = 0;
			if (systemStatus & GHOST_ON) data._ghostY = updateGhostLocation(field, tetromino, loc);
			
			spools.push(data); // スプールにセッションデータを保存
			if (!(systemStatus & WAITING)) {
				systemStatus |= WAITING;
				drawLocation = 0;
				nextBoard.addEventListener(Event.ENTER_FRAME, nextBoardOperate);
			}
			
			// ゲームオーバー判定の結果を返す。ゲームオーバーじゃなければ真、ゲームオーバーなら偽
			return (!first && overlapChecker(field, tetromino, loc))? false: true;
			
		} // setTetromino() close
		
		
		//------------------------------------------------------------
		// Next Tetromino
		//
		// 次のテトリミノの為の処理
		//------------------------------------------------------------
		private function nextBoardOperate(event:Event):void {
			/**
			 * 機能：次のテトリミノの表示オブジェクトの描画を操作する。
			 *      描画ループ完了後、フィールドに新しいテトリミノを配置する。
			 * 引数：イベントオブジェクト
			 * 戻り値：なし
			 * 備考：
			 */
			
			drawNextBoard(drawLocation); // 描画
			drawLocation -= 10;          // 次の描画開始位置をずらす
			
			// 描画ループ完了
			if (drawLocation <= -LOC_INTERVAL) {
				// リスナー関数を削除
				nextBoard.removeEventListener(Event.ENTER_FRAME, nextBoardOperate);
				
				// 新しいテトリミノコードを用意する
				for (var i:uint = 1; i < NUM_NEXT; i++) {
					nextCodes[i - 1] = nextCodes[i] as uint; // データを前へ詰める
				}
				nextCodes[NUM_NEXT - 1] = Math.floor(Math.random() * 7) + 1; // 末尾に新しいコードを入れる
				
				// 次のテトリミノ待ちのセッションデータをスプールから取り出す
				var data:DataStructure = spools[0] as DataStructure;
				spools[0] = null;
				spools.splice(0, 1);
				
				if (data._status & GAME_OVER) {
					// ゲームオーバー
					data._timer.removeEventListener(TimerEvent.TIMER, data._dropHandler);
					fieldMapping(data); // 重なった状態で反映される
					onGameOver(data);   // ゲームオーバー処理実行
					systemStatus &= ~GAME_START;
					systemStatus |= GAME_OVER;
				} else {
					// ゲームオーバーではない
					fieldMapping(data); // 新しいテトリミノをフィールドに反映
					drawField(data);    // フィールドの描画
					stage.addEventListener(KeyboardEvent.KEY_DOWN, data._keyHandler);
					data._timer.start();
				}
				data = null;
				
				// スプールを確認して次のテトリミノを待っているセッションがあるかどうかで分岐
				if (spools.length > 0) {
					drawLocation = 0;
					nextBoard.addEventListener(Event.ENTER_FRAME, nextBoardOperate);
				} else {
					systemStatus &= ~WAITING;
				}
			}
			
		} // nextBoardOperate() close
		
		
		private function drawNextBoard(startY:int):void {
			/**
			 * 機能：次のテトリミノ表示オブジェクトに描画を実行する。
			 * 引数：描画を開始する基準位置座標
			 * 戻り値：なし
			 * 備考：描画内容は必ずnextCodesのデータ内容に基ずき、描画対象は必ずdrawBoard
			 *      それ以外はdrawField()とほぼ同じ
			 */
			
			var x:int, y:int, px:int, py:int, reviseX:int, len:uint, num:uint, code:int, pattern:Vector.<Vector.<uint>>;
			
			drawBoard.graphics.clear();
			
			nextCodes.forEach(
			function (element:uint, index:int, array:Vector.<uint>):void {
				pattern = tetriminoInit(element);             // 描画するパターンデータを取得
				len     = pattern.length;                     // パターンデータの配列長を取得
				py      = index * LOC_INTERVAL + startY + 10; // 描画開始するy座標
				reviseX = -len * CELL_SIZE * 0.5;             // 描画開始するx座標
				
				if (len > 3) py -= CELL_SIZE * 0.5; // 棒型の場合はちょっと上にずらす
				
				for (y = 0; y < len; y++) {
					px = reviseX;
					for (x = 0; x < len; x++) {
						code = pattern[y][x];
						if (code > EMPTY) {
							drawBoard.graphics.beginFill(colorInit(code));
							drawBoard.graphics.lineStyle(1, LINE_COLOR);
							drawBoard.graphics.drawRect(px, py, CELL_SIZE, CELL_SIZE);
							drawBoard.graphics.endFill();
						}
						px += CELL_SIZE;
					}
					py += CELL_SIZE;
				}
			}
			);
			
			drawBoard.graphics.endFill();
			
		} // drowNextBoard() close
		
		
		
//================================================================================
// Tetromino Control
//
// 主に落下してくるテトリミノの操作に関わる処理
//================================================================================
		private function keyHandlerSet(data:DataStructure):void {
			/**
			 * 機能：セッション別にキーボードイベントを設定する
			 * 引数：イベントを設定したいセッションのデータ構造体
			 * 戻り値：なし
			 * 備考：リスナー関数をクロージャにする事で、データ構造体を保持させつつ、
			 *      そのデータ構造体自体のプロパティにしておく
			 */
			
			data._keyHandler = function(event:KeyboardEvent):void {
				switch(event.keyCode) {
					case Keyboard.LEFT:  tetrominoControl(Keyboard.LEFT,  data); break; // 左
					case Keyboard.RIGHT: tetrominoControl(Keyboard.RIGHT, data); break; // 右
					case Keyboard.DOWN:  tetrominoControl(Keyboard.DOWN,  data); break; // 下
					case Keyboard.UP:    tetrominoControl(Keyboard.UP,    data); break; // 回転
					case Keyboard.SPACE: getHardLocation(data);                  break; // ハードドロップ
				}
			}
			stage.addEventListener(KeyboardEvent.KEY_DOWN, data._keyHandler);
			
		} // onKeyDownHandler() close
		
		
		private function tetrominoControl(direct:uint, data:DataStructure):Boolean {
			/**
			 * 機能：指示に合わせてテトリミノを動かす。引数でとったコードに該当する処理を実行する。
			 * 引数：第1引数 テトリミノを動かす方向指示コード（基本的にはキーコード）、第2引数 プレイヤーデータ構造体
			 * 戻り値：移動方向に何かあるかどうかの真偽値。何かあった場合に真を返す。
			 * 備考：
			 */
			
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			switch (direct) {
				case Keyboard.LEFT: // 左
					if (getLeftHit(data)) return true; // 真が返る（左に何かある）と処理を終了
					
					fieldMapping(data, CLEAR); // 動かせる場合はフィールドデータから直前のテトリミノデータを削除
					loc.x--;                   // 位置情報の更新
					
					// 棒形縦、左壁際での補正。配列の縮小
					if (tetromino.length > 3 && loc.x < 0) sizeReduction(tetromino, loc);
					
					// ゴーストブロック座標更新
					if (systemStatus & GHOST_ON) data._ghostY = updateGhostLocation(field, tetromino, loc);
					
					fieldMapping(data); // フィールドデータにテトリミノの新しい位置情報を反映
					drawField(data);    // フィールドを再描画
					
					return false;
					
				case Keyboard.RIGHT: // 右
					if (getRightHit(data)) return true; // 真が返る（右に何かある）と処理を終了
					
					fieldMapping(data, CLEAR); // 動かせる場合はフィールドデータから直前のテトリミノデータを削除
					loc.x++;                   // 位置情報の更新
					
					// 棒形縦、左壁際での補正。配列の復元
					if (tetromino.length > 3 && loc.x == 1) restoreSize(tetromino, loc);
					
					// ゴーストブロック座標更新
					if (systemStatus & GHOST_ON) data._ghostY = updateGhostLocation(field, tetromino, loc);
					
					fieldMapping(data); // フィールドデータにテトリミノの新しい位置情報を反映
					drawField(data);    // フィールドを再描画
					
					return false;
					
				case Keyboard.DOWN: // 下
					data._timer.stop(); // 自動落下タイマーをいったん停止
					
					if (getBottomHit(data)) { // 下にぶつかった場合
						// 一時的にキー操作イベントを削除する
						stage.removeEventListener(KeyboardEvent.KEY_DOWN, data._keyHandler);
						
						// 自動落下速度を調整
						dropTimerRegulator(data);
						
						fieldMapping(data, FIXED); // フィールドデータにテトリミノコードを固定する
						var lines:Vector.<uint> = lineChecker(field); // 揃っているラインがあるかどうかを問い合わせる
						
						// 揃ったラインがあった場合
						if (lines.length > 0) {
							// 揃っているラインを一時的に空白行にして描画する。
							fieldMapping(data, CLEAR, lines);
							drawField(data);
							
							// ラインが揃って空白を詰める処理を少し遅らせて実行
							var ragTimer:Timer = new Timer(FILL_RAG, 1);
							ragTimer.addEventListener(TimerEvent.TIMER,
								function(event:TimerEvent):void { // 無名関数の登録
									ragTimer.removeEventListener(TimerEvent.TIMER, arguments.callee);
									
									fieldMapping(data, FILL, lines); // 空白のラインを詰める
									baseClear(field, lines.length);  // 固定フィールドより上の処理
									drawField(data);
									
									// 新しいテトリミノデータを用意
									if (setTetromino(data)) {
										// ゲームオーバーではない
										if (lines.length >= 4) dropTimerRegulator(data, true);
										data._status &= ~GAME_OVER;
									} else {
										// ゲームオーバー
										data._timer.removeEventListener(TimerEvent.TIMER, data._dropHandler);
										data._status |= GAME_OVER;
									}
									
									ragTimer = null;
								});
							ragTimer.start();
							
							return true; // 処理終了
						}
						
						// 揃ったラインが無い場合。新しいテトリミノデータを用意
						if (setTetromino(data)) {
							// ゲームオーバーではない
							data._status &= ~GAME_OVER;
						} else {
							// ゲームオーバー
							data._timer.removeEventListener(TimerEvent.TIMER, data._dropHandler);
							data._status |= GAME_OVER;
						}
						
						return true; // 処理終了
					}
					
					// ぶつからなかった場合（動かせる）
					if (data._status & HARDDROP_ON) data._status &= ~HARDDROP_ON; // ハードドロップフラグを偽
					else fieldMapping(data, CLEAR);                               // フィールドデータから直前のテトリミノデータを削除
					loc.y++;             // 位置情報の更新
					fieldMapping(data);  // フィールドデータにテトリミノの新しい位置情報を反映
					drawField(data);     // フィールドを再描画
					data._timer.start(); // 自動落下タイマースタート
					
					return false;
					
				case Keyboard.UP: // 回転
					if (getRotateHit(data)) return true; // 真が返ると処理を終了
					
					fieldMapping(data, CLEAR); // 回転させる前に直前のテトリミノデータを削除
					
					// 回転補正
					if (loc.x == 0) {
						loc.x++; // loc.x:0の場合場合、loc.x++で確定。
						// 棒形例外用。復元処理。Pointはダミー。
						if (tetromino.length - tetromino[0].length != 0) restoreSize(tetromino, new Point(0, 0));
					}
					// loc.x+パターンデータ幅がフィールド幅と同じ場合、loc.x--で確定。
					if (loc.x + tetromino[0].length == FIELD_LEN_W) loc.x--;
					// loc.x+パターンデータ幅がフィールド幅より大きい場合、loc.x-=2で確定。棒形例外用。
					if (loc.x + tetromino[0].length > FIELD_LEN_W)  loc.x -= 2;
					
					tetrominoRotate(tetromino); // テトリミノパターンデータを90度回転
					
					// ゴーストブロック座標更新
					if (systemStatus & GHOST_ON) data._ghostY = updateGhostLocation(field, tetromino, loc);
					
					fieldMapping(data); // フィールドデータにテトリミノの新しい位置情報を反映
					drawField(data);    // フィールドを再描画
					
					return false;
			}
			
			return false;
			
		} // tetrominoControl() close
		
		
		//------------------------------------------------------------
		// Data Control Tools
		//
		// テトリミノのパターンデータ配列自体を操作する時に使う
		//------------------------------------------------------------
		private function tetrominoRotate(tetromino:Vector.<Vector.<uint>>):Vector.<Vector.<uint>> {
			/**
			 * 機能：テトリミノのパターンデータを90度回転する
			 * 引数：回転させたいテトリミノのパターンデータ配列
			 * 戻り値：回転させたパターンデータ配列
			 * 備考：
			 */
			
			var x:uint, y:uint,
					xSize:uint = tetromino[0].length,
					ySize:uint = tetromino.length,
					patternBuff:Vector.<Vector.<uint>>;
			
			patternBuff = new Vector.<Vector.<uint>>(xSize);
			for (y = 0; y < ySize; y++) patternBuff[y] = tetromino[y].concat();
			for (y = 0; y < ySize; y++) {
				for (x = 0; x < xSize;  x++) tetromino[x][ySize - 1 - y] = patternBuff[y][x];
			}
			
			return tetromino;
			
		} // tetrominoRotate() close
		
		private function sizeReduction(tetromino:Vector.<Vector.<uint>>, loc:Point):Vector.<Vector.<uint>> {
			/**
			 * 機能：位置情報のx値が負になった場合に、
			 *      落下中のテトリミノパターンデータ配列の二次元目の配列長を要素一つ分縮小する。
			 * 引数：第1引数 縮小したいテトリミノのパターンデータ配列
			 *      第2引数 落下中のテトリミノの位置情報Point
			 * 戻り値：縮小したテトリミノのパターンデータ配列
			 * 備考：この処理への評価を通る状況は、
			 *      棒形テトリミノが特定の配列状態で、左の壁に接近した場合に限定される。restoreSize()と対と考える。
			 */
			
			tetromino.forEach(
				function (element:Vector.<uint>, index:int, array:Vector.<Vector.<uint>>):void {
					element.reverse(); element.pop();
				}
			);
			loc.x = 0;
			return tetromino;
			
		} // sizeReduction() close
		
		private function restoreSize(tetromino:Vector.<Vector.<uint>>, loc:Point):Vector.<Vector.<uint>> {
			/**
			 * 機能：落下中のテトリミノのパターンデータ配列の一次元目と二次元目の配列長が違っている場合にそれを同じ長さに復元する。
			 * 引数：復元したいテトリミノのパターンデータ配列
			 * 戻り値：復元したテトリミノのパターンデータ他配列
			 * 備考：この処理の評価を通る状況はsizeReduction()メソッドで縮小した次の動作で、
			 * 　　　　パターンデータ配列が右に動いた直後に限定される。sizeReduction()と対と考える。
			 */
			
			var xSize:uint = tetromino[0].length, ySize:uint = tetromino.length;
			if (ySize - xSize != 0) {
				tetromino.forEach(
					function (element:Vector.<uint>, index:int, array:Vector.<Vector.<uint>>):void {
						element.push(EMPTY); element.reverse();
					}
				);
				loc.x--;
			}
			return tetromino;
			
		} // restreSize() clsoe
		
		
		//------------------------------------------------------------
		// Tetromino Hit Check
		//
		// 機能：テトリミノパターン配列内のテトリミノコードが入った要素を探し、
		//      検出した場合はその要素のフィールドでの位置から隣接要素を参照して何かあれば真を返す。
		// 引数：プレイヤーデータ構造体
		// 戻り値：問い合わせられた方向に対して何かを検出した場合は真を返す
		// 備考：壁もしくは固定されたテトリミノのコードを検出すると真を返して終了
		//------------------------------------------------------------
		private function getLeftHit(data:DataStructure):Boolean {
			// 左方向へのヒットチェック
			
			var x:int, y:int, fx:int, fy:int, xSize:uint, ySize:uint;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			xSize = tetromino[0].length,
			ySize = tetromino.length;
			
			for (x = 0; x < xSize; x++) { // 左から右へ走査
			for (y = 0; y < ySize; y++) { // 上から下へ走査
				if (tetromino[y][x]) {
					fx = loc.x + x; // テトリミノコードが入ったセルの、フィールド上でのx位置
					fy = loc.y + y; // テトリミノコードが入ったセルの、フィールド上でのy位置
					if (field[fy][fx - 1] > FIXED) return true;
				}
			}
			}
			return false;
			
		} // getLeftHit() close
		
		private function getRightHit(data:DataStructure):Boolean {
			// 右方向へのヒットチェック
			
			var x:int, y:int, fx:int, fy:int, xSize:uint, ySize:uint;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			xSize = tetromino[0].length,
			ySize = tetromino.length;
			
			for (x = xSize - 1; x >= 0; x--) { // 右から左へ走査
			for (y = 0; y < ySize; y++) {      // 上から下へ走査
				if (tetromino[y][x]) {
					fx = loc.x + x; // テトリミノコードが入ったセルの、フィールド上でのx位置
					fy = loc.y + y; // テトリミノコードが入ったセルの、フィールド上でのy位置
					if (field[fy][fx + 1] > FIXED) return true;
				}
			}
			}
			return false;
			
		} // getRightHit() close
		
		private function getBottomHit(data:DataStructure):Boolean {
			// 下方向へのヒットチェック
			
			var x:int, y:int, fx:int, fy:int, xSize:uint, ySize:uint;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			xSize = tetromino[0].length,
			ySize = tetromino.length;
			
			for (y = ySize - 1; y >= 0; y--) { // 下から上へ走査
			for (x = 0; x < xSize; x++) {      // 左から右へ走査
				if (tetromino[y][x]) {
					fx = loc.x + x; // テトリミノコードが入ったセルの、フィールド上でのx位置
					fy = loc.y + y; // テトリミノコードが入ったセルの、フィールド上でのy位置
					if (field[fy + 1][fx] > FIXED) return true;
				}
			}
			}
			return false;
			
		} // getBottomHit() close
		
		private function getRotateHit(data:DataStructure):Boolean {
			/**
			 * 機能：テトリミノパターン配列のデータのコピーを90度回転させた上でテトリミノコードが入った要素を探し、
			 *      コードを検出した場合はその要素のフィールドでの位置の要素を参照して何かあれば真を返す。
			 * 引数：プレイヤーデータ構造体
			 * 戻り値：真偽値
			 * 備考：回転させてシミュレーションを行う。この処理内ではあくまでもシミュレートに限定。
			 *      ホントのパターン配列、位置情報に対しては何も行わない事。
			 */
			
			var x:int, y:int, fx:int, fy:int, revise:int = 0, xSize:uint, ySize:uint, patternBuff:Vector.<Vector.<uint>>;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			xSize = tetromino[0].length,
			ySize = tetromino.length;
			
			// 一度コピーを作成してから、パターンデータを90度回転させる
			patternBuff = new Vector.<Vector.<uint>>(xSize);
			for (y = 0; y < ySize; y++) patternBuff[y] = tetromino[y].concat();
			xSize = patternBuff[0].length;
			ySize = patternBuff.length;
			
			// 回転補正　左の壁際
			if (loc.x == 0) {
				revise = 1;
				// 棒形例外用。
				if (xSize - ySize != 0) {
					restoreSize(patternBuff, new Point(0, 0)); // 復元処理。ダミーのPointを渡す。
					xSize = patternBuff[0].length;
					ySize = patternBuff.length;
				}
			}
			// 回転補正 右の壁際
			if (loc.x + xSize == FIELD_LEN_W) revise = -1;
			if (loc.x + xSize > FIELD_LEN_W)  revise = -2; // 棒形例外用
			
			tetrominoRotate(patternBuff); // パターンデータを回転
			
			// フィールド上のセルを参照して何かと重なっていないかを調べて、戻り値（真偽値）をそのまま返す
			return overlapChecker(field, patternBuff, new Point(loc.x + revise, loc.y));
			
		} // getRotateHit() close
		
		private function getHardLocation(data:DataStructure):Boolean {
			/**
			 * 機能：ハードドロップが実行された際に、ハードドロップ用のフィールド座標を探索し、
			 *      dropLocationの値を変更する。
			 *      その後、tetrominoControl()に下移動として処理を渡す仲介処理。
			 * 引数：データ構造体
			 * 戻り値：真偽値
			 * 備考：
			 */
			
			var x:int, y:int;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			if (systemStatus & GHOST_ON) {
				// ゴーストブロックがオンになっている場合は、その座標を利用する
				if (data._ghostY - 1 >= loc.y) {
					fieldMapping(data, CLEAR);             // フィールドデータから直前のテトリミノデータを削除
					loc.y = data._ghostY - 1;              // ゴーストブロックの座標からハードドロップの座標を算出
					data._status |= HARDDROP_ON;           // ハードドロップフラグオン
					tetrominoControl(Keyboard.DOWN, data); // 下移動のコードを渡して実行
					return true;
				}
				
			} else {
				// 通常の探索処理
				for (y = loc.y + 1; y < FIELD_LEN_H - 1; y++) {
					if (overlapChecker(field, tetromino, new Point(loc.x, y))) {
						if (y - 2 >= loc.y) {
							fieldMapping(data, CLEAR);             // フィールドデータから直前のテトリミノデータを削除
							loc.y = y - 2;                         // フィールド座標y値を変更（最下部より二つ手前）
							data._status |= HARDDROP_ON;           // ハードドロップフラグオン
							tetrominoControl(Keyboard.DOWN, data); // 下移動のコードを渡して実行
							return true;
						} else {
							// 最下部のヒット判定が返っても、更新後の座標が現在の座標よりも上になるなら何もしない
							return false;
						}
					}
				}
			}
			return false;
			
		} // getHardLocation() close
		
		
		
//================================================================================
// Field Operater
//
// 主にフィールドデータにかかわる処理や描画
//================================================================================
		private function fieldMapping(
				data:DataStructure,        // プレイヤーデータ構造体
				type:uint = DROP,          // テトリミノコードをどう処理するかの処理指示コード
				lines:Vector.<uint> = null // フィールドデータ上の、揃っているラインの行数番号の配列
			):void {
			/**
			 * 機能：フィールドデータ配列に、出現したテトリミノのパターンデータを反映する。
			 * 引数：宣言部に記載。下は第4引数の指示コードについて。
			 *      DROP:フィールドデータに反映 CLEAR:フィールドデータから削除
			 *      FIXED:フィールドデータに固定 FILL:揃ったラインを削除して詰める
			 * 戻り値：なし
			 * 備考：第5引数がnullじゃない時にtypeがFILL、CLEAR以外だとエラーになる。でもtryは付けないので注意
			 */
			
			var x:uint, y:uint, ex:uint, ey:uint, px:uint, py:uint, code:uint, index:uint, xSize:uint, ySize:uint;
			var field:Vector.<Vector.<int>>      = data._field;
			var tetromino:Vector.<Vector.<uint>> = data._tetromino;
			var loc:Point                        = data._location;
			
			xSize = tetromino[0].length;
			ySize = tetromino.length;
			
			// 揃ったラインのデータがあるかどうかで分岐
			if (lines) {
				ey = lines.length;
				for (y = 0; y < ey; y++) {
					if (type == CLEAR) { // 揃っているラインを空白データに差し替える
						field.splice(lines[y], 1, Vector.<int>([WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,WALL]));
					}
					if (type == FILL)  {
						field.splice(lines[y], 1); // 揃ったライン（空白）の配列データ削除
						// 固定フィールドの最上段に空白データを入れ込む
						field.splice(2, 0, Vector.<int>([WALL,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,EMPTY,WALL]));
					}
				}
			} else {
				ex = loc.x + xSize; // x軸走査の終了位置
				ey = loc.y + ySize; // y軸走査の終了位置
				for (y = loc.y; y < ey; y++) {
				for (x = loc.x; x < ex; x++) {
					px = x - loc.x; // テトリミノパターンデータ配列用のインデックス
					py = y - loc.y; // テトリミノパターンデータ配列用のインデックス
					code = tetromino[py][px];
					if (code) {
						switch (type) {
							case CLEAR: // 削除
								if (systemStatus & GHOST_ON) field[data._ghostY + py][x] = EMPTY;
								field[y][x] = EMPTY;
								break;
								
							case FIXED: // 固定
								field[y][x] = code + FIXED;
								break;
								
							default: // 落下中
								if (systemStatus & GHOST_ON) field[data._ghostY + py][x] = GHOST;
								field[y][x] = code;
						}
					}
				}
				}
			}
			
		} // fieldMapping() close
		
		
		private function drawField(data:DataStructure):void {
			/**
			 * 機能：フィールドデータの情報に基ずいて、表示コンテナにフィールドを描画する。
			 * 引数：プレイヤーデータ構造体
			 * 戻り値：なし
			 * 備考：
			 */
			
			var x:int, y:int, px:int, py:int, code:int;
			var field:Vector.<Vector.<int>> = data._field;
			var container:Shape             = data._container;
			
			container.graphics.clear();
			
			for (y = 0; y < FIELD_LEN_H; y++) {
				px = 0;
				for (x = 0; x < FIELD_LEN_W; x++) {
					code = field[y][x];
					if (code > EMPTY) {
						if (code < WALL && code > FIXED) code -= FIXED; // 固定コードの復元
						container.graphics.beginFill(colorInit(code));
						container.graphics.lineStyle(1, LINE_COLOR);
						container.graphics.drawRect(px, py, CELL_SIZE, CELL_SIZE);
						container.graphics.endFill();
					}
					px += CELL_SIZE;
				}
				py += CELL_SIZE;
			}
			
			container.graphics.endFill();
			
		} // drawField() close
		
		
		//------------------------------------------------------------
		// Field Search Tools
		//------------------------------------------------------------
		private function overlapChecker(
				field:Vector.<Vector.<int>>,      // Vector二次元配列 フィールドデータ配列
				tetromino:Vector.<Vector.<uint>>, // Vector二次元配列 調べたいテトリミノのパターンデータ配列
				loc:Point                         // 調べたいフィールド上の座標
			):Boolean {
			/**
			 * 機能：テトリミノデータが、任意の座標において、固定されたテトリミノ及び壁と重なるかどうかを調べて真偽値を返す
			 * 引数：宣言部に記載
			 * 戻り値：重なる場合は真を返す。
			 * 備考：
			 */
			
			var x:uint, y:uint, fx:uint, fy:uint,
					xSize:uint = tetromino[0].length,
					ySize:uint = tetromino.length;
			
			for (y = 0; y < ySize; y++) {
			for (x = 0; x < xSize; x++) {
				if (tetromino[y][x]) {
					fx = loc.x + x; // テトリミノコードが入ったセルの、フィールド上でのx位置
					fy = loc.y + y; // テトリミノコードが入ったセルの、フィールド上でのy位置
					if (field[fy][fx] > FIXED) return true;
				}
			}
			}
			return false;
			
		} // overlapChecker() close
		
		
		private function lineChecker(field:Vector.<Vector.<int>>):Vector.<uint> {
			/**
			 * 機能：フィールド上に揃ったラインがあるかどうかを探索し、揃った行があった場合はそのラインの行数を配列で返す
			 * 引数：フィールドデータ配列
			 * 戻り値：揃っているラインの行数情報を入れた配列
			 * 備考：
			 */
			
			var x:uint, y:int, emptyCount:int = 1, filled:Boolean, lines:Vector.<uint>;
			
			lines = new Vector.<uint>();
			
			for (y = FIELD_LEN_H - 2; y >= 2 && emptyCount > EMPTY; y--) {
				// 固定フィールドの下のラインから、空白のセルがあるかどうかを調べる
				filled = true;
				for (x = 0; x < FIELD_LEN_W && filled; x++) if (field[y][x] == EMPTY) filled = false;
				
				if (filled) {
					lines.push(y); // 空白の無いラインがあった場合は、そのラインのインデックスを保存
				} else {
					// ラインが揃っていないことがわかった場合は、そのラインが完全な空白行かどうかを調べる。
					// 空白行の場合はそれより上の行にテトリミノがある事がないので、探索ループを終了する。
					emptyCount = WALL * (-2); // 壁の値を減算しておく
					for (x = 0; x < FIELD_LEN_W; x++) emptyCount += field[y][x];
				}
			}
			
			return lines.reverse();
			
		} // lineChecker() close
		
		
		private function baseClear(field:Vector.<Vector.<int>>, len:uint):void {
			/**
			 * 機能：固定フィールドより上のセルに壁以外のコードが入っていた場合に、消えたラインの分だけ値を下へ移動する。
			 * 引数：第1引数 フィールドデータ配列、第2引数 揃ったラインの数
			 * 戻り値：なし
			 * 備考：
			 */
			
			var x:uint, y:int;
			for (y = 1; y >= 0; y--) {
			for (x = 3; x < 9;  x++) {
				if (field[y][x] > EMPTY) {
					field[y + len][x] = field[y][x] as int;
					field[y][x] = EMPTY;
				}
			}
			}
			
		} // baseClear() close
		
		
		private function updateGhostLocation(
				field:Vector.<Vector.<int>>,      // Vector二次元配列 フィールドデータ配列
				tetromino:Vector.<Vector.<uint>>, // Vector二次元配列 調べたいテトリミノのパターンデータ配列
				loc:Point                         // 探索を開始する座標
			):int {
			/**
			 * 機能：ゴーストブロックを描画するy座標を更新する
			 * 引数：プレイヤーデータ構造体
			 * 戻り値：更新したかどうかの真偽値。更新した場合は真を返す。
			 * 備考：末尾のreturnはダミー
			 */
			
			var y:int;
			
			for (y = loc.y + 1; y < FIELD_LEN_H - 1; y++) {
				if (overlapChecker(field, tetromino, new Point(loc.x, y))) return y - 1;
			}
			return 0;
			
		} // updateGhostLocation() close
	}
}